import { __read } from "tslib";
var DOMAIN_PATTERN = /^[a-z0-9][a-z0-9\.\-]{1,61}[a-z0-9]$/;
var IP_ADDRESS_PATTERN = /(\d+\.){3}\d+/;
var DOTS_PATTERN = /\.\./;
export var DOT_PATTERN = /\./;
export var S3_HOSTNAME_PATTERN = /^(.+\.)?s3(-fips)?(\.dualstack)?[.-]([a-z0-9-]+)\./;
var S3_US_EAST_1_ALTNAME_PATTERN = /^s3(-external-1)?\.amazonaws\.com$/;
var AWS_PARTITION_SUFFIX = "amazonaws.com";
export var isBucketNameOptions = function (options) { return typeof options.bucketName === "string"; };
export var isDnsCompatibleBucketName = function (bucketName) {
    return DOMAIN_PATTERN.test(bucketName) && !IP_ADDRESS_PATTERN.test(bucketName) && !DOTS_PATTERN.test(bucketName);
};
var getRegionalSuffix = function (hostname) {
    var parts = hostname.match(S3_HOSTNAME_PATTERN);
    return [parts[4], hostname.replace(new RegExp("^".concat(parts[0])), "")];
};
export var getSuffix = function (hostname) {
    return S3_US_EAST_1_ALTNAME_PATTERN.test(hostname) ? ["us-east-1", AWS_PARTITION_SUFFIX] : getRegionalSuffix(hostname);
};
export var getSuffixForArnEndpoint = function (hostname) {
    return S3_US_EAST_1_ALTNAME_PATTERN.test(hostname)
        ? [hostname.replace(".".concat(AWS_PARTITION_SUFFIX), ""), AWS_PARTITION_SUFFIX]
        : getRegionalSuffix(hostname);
};
export var validateArnEndpointOptions = function (options) {
    if (options.pathStyleEndpoint) {
        throw new Error("Path-style S3 endpoint is not supported when bucket is an ARN");
    }
    if (options.accelerateEndpoint) {
        throw new Error("Accelerate endpoint is not supported when bucket is an ARN");
    }
    if (!options.tlsCompatible) {
        throw new Error("HTTPS is required when bucket is an ARN");
    }
};
export var validateService = function (service) {
    if (service !== "s3" && service !== "s3-outposts" && service !== "s3-object-lambda") {
        throw new Error("Expect 's3' or 's3-outposts' or 's3-object-lambda' in ARN service component");
    }
};
export var validateS3Service = function (service) {
    if (service !== "s3") {
        throw new Error("Expect 's3' in Accesspoint ARN service component");
    }
};
export var validateOutpostService = function (service) {
    if (service !== "s3-outposts") {
        throw new Error("Expect 's3-posts' in Outpost ARN service component");
    }
};
export var validatePartition = function (partition, options) {
    if (partition !== options.clientPartition) {
        throw new Error("Partition in ARN is incompatible, got \"".concat(partition, "\" but expected \"").concat(options.clientPartition, "\""));
    }
};
export var validateRegion = function (region, options) {
    if (region === "") {
        throw new Error("ARN region is empty");
    }
    if (options.useFipsEndpoint) {
        if (!options.allowFipsRegion) {
            throw new Error("FIPS region is not supported");
        }
        else if (!isEqualRegions(region, options.clientRegion)) {
            throw new Error("Client FIPS region ".concat(options.clientRegion, " doesn't match region ").concat(region, " in ARN"));
        }
    }
    if (!options.useArnRegion &&
        !isEqualRegions(region, options.clientRegion || "") &&
        !isEqualRegions(region, options.clientSigningRegion || "")) {
        throw new Error("Region in ARN is incompatible, got ".concat(region, " but expected ").concat(options.clientRegion));
    }
};
export var validateRegionalClient = function (region) {
    if (["s3-external-1", "aws-global"].includes(region)) {
        throw new Error("Client region ".concat(region, " is not regional"));
    }
};
var isEqualRegions = function (regionA, regionB) { return regionA === regionB; };
export var validateAccountId = function (accountId) {
    if (!/[0-9]{12}/.exec(accountId)) {
        throw new Error("Access point ARN accountID does not match regex '[0-9]{12}'");
    }
};
export var validateDNSHostLabel = function (label, options) {
    if (options === void 0) { options = { tlsCompatible: true }; }
    if (label.length >= 64 ||
        !/^[a-z0-9][a-z0-9.-]*[a-z0-9]$/.test(label) ||
        /(\d+\.){3}\d+/.test(label) ||
        /[.-]{2}/.test(label) ||
        ((options === null || options === void 0 ? void 0 : options.tlsCompatible) && DOT_PATTERN.test(label))) {
        throw new Error("Invalid DNS label ".concat(label));
    }
};
export var validateCustomEndpoint = function (options) {
    if (options.isCustomEndpoint) {
        if (options.dualstackEndpoint)
            throw new Error("Dualstack endpoint is not supported with custom endpoint");
        if (options.accelerateEndpoint)
            throw new Error("Accelerate endpoint is not supported with custom endpoint");
    }
};
export var getArnResources = function (resource) {
    var delimiter = resource.includes(":") ? ":" : "/";
    var _a = __read(resource.split(delimiter)), resourceType = _a[0], rest = _a.slice(1);
    if (resourceType === "accesspoint") {
        if (rest.length !== 1 || rest[0] === "") {
            throw new Error("Access Point ARN should have one resource accesspoint".concat(delimiter, "{accesspointname}"));
        }
        return { accesspointName: rest[0] };
    }
    else if (resourceType === "outpost") {
        if (!rest[0] || rest[1] !== "accesspoint" || !rest[2] || rest.length !== 3) {
            throw new Error("Outpost ARN should have resource outpost".concat(delimiter, "{outpostId}").concat(delimiter, "accesspoint").concat(delimiter, "{accesspointName}"));
        }
        var _b = __read(rest, 3), outpostId = _b[0], _1 = _b[1], accesspointName = _b[2];
        return { outpostId: outpostId, accesspointName: accesspointName };
    }
    else {
        throw new Error("ARN resource should begin with 'accesspoint".concat(delimiter, "' or 'outpost").concat(delimiter, "'"));
    }
};
export var validateNoDualstack = function (dualstackEndpoint) {
    if (dualstackEndpoint)
        throw new Error("Dualstack endpoint is not supported with Outpost or Multi-region Access Point ARN.");
};
export var validateNoFIPS = function (useFipsEndpoint) {
    if (useFipsEndpoint)
        throw new Error("FIPS region is not supported with Outpost.");
};
export var validateMrapAlias = function (name) {
    try {
        name.split(".").forEach(function (label) {
            validateDNSHostLabel(label);
        });
    }
    catch (e) {
        throw new Error("\"".concat(name, "\" is not a DNS compatible name."));
    }
};
